import os
import logging
import logging.config
import yaml
from .default_log_conf import DEFAULT_LOG_CONF


class PathUtil:
    __project_root = None

    @staticmethod
    def init(project_root):
        if PathUtil.__project_root is not None:
            raise ValueError('PathUtil only could init once')
        PathUtil.__project_root = project_root

    @staticmethod
    def get_abs_path(relative_path):
        """
        Return the absolute path of PROJECT_ROOT join relative_path
        default PROJECT_ROOT is None, you must set it by using init() method.
        """
        if PathUtil.__project_root is None:
            raise ValueError('Please init PathUtil before use it.')
        if os.path.isabs(relative_path):
            return relative_path
        return os.path.join(PathUtil.__project_root, relative_path)


class ConfigUtil:

    @staticmethod
    def load_config(conf_path):
        """
        解析yaml配置文件
        Args:
            conf_file:yaml配置文件路径
        Returns:
        解析后的配置对象
        """
        conf_path = ConfigUtil.__get_conf_path(conf_path)
        with open(conf_path, 'r', encoding='utf-8') as conf:
            conf = yaml.load(conf, yaml.SafeLoader)
        return conf

    @staticmethod
    def __get_conf_path(conf_path):
        """
        Find config file in all sys.path + relative_file_path.
        Return the absolute path of first match path
        """
        if os.path.isabs(conf_path):
            return conf_path
        file_path = PathUtil.get_abs_path(conf_path)
        if os.path.isfile(file_path):
            return file_path
        raise ValueError('Config file [{}] not found.'.format(file_path))


class LogUtil:
    __log_conf = None

    @staticmethod
    def get_logger(name):
        if LogUtil.__log_conf is None:
            try:
                conf = ConfigUtil.load_config('conf/logging.yaml')
                LogUtil._format_conf(conf)
                LogUtil.__log_conf = conf
                print(LogUtil.__log_conf)
            except ValueError:
                LogUtil.__log_conf = DEFAULT_LOG_CONF
            logging.config.dictConfig(LogUtil.__log_conf)
        return logging.getLogger(name)

    @staticmethod
    def _format_conf(conf):
        for k, v in conf.items():
            if isinstance(v, dict):
                LogUtil._format_conf(v)
            else:
                if k == 'filename':
                    conf[k] = PathUtil.get_abs_path(v)
