#! /usr/bin/env python
# -*- coding: utf-8 -*_
# Author: Liu Yang <mkliuyang@gmail.com>
"""
config 模块
===============

这是一个处理 config 的模块，模块支持`类与函数的注入式配置`，`增加配置`，`继承配置（多继承）`，`配置文件`，`多指令执行`。
相当于兼容 `python.ConfigParser` + `python.argparse` + `google:python-fire` 的库。

1. 添加配置项

::

    from dlab.config import add_config
    @add_config('config_name1', int, default=1, desc='help information for config_name1')
    @add_config('config_name2', str, default='', desc='help information for config_name1')
    def func_1(config):
        print(config['config_name1'], config['config_name2']) # 通过名字来进行访问，config这个参数是调用时自动注入的。

    @inherit_config(func_1)
    @add_config('config_name3', int, default=1, desc='help information for config_name3')
    def func_2(config):
        func_1()  # config这个参数是调用时自动注入的。在调用时不需要传入。
        print(config['config_name1'], config['config_name2'])  # 继承子过程可以调用到父过程的配置，反之不能。
        print(config['config_name3'])

2. 使用 `fire` 来执行

in main.py

::

    from dlab.config import fire
    fire([func_1, func_2])  # 只有注册过参数的过程才能被调用，这个过程应只有一个参数。

in bash

::

    $ main.py --help # 查看全局help
    usage: main.py [-h] [--cfg CFG] [--save_cfg SAVE_CFG]
                      {func_1,func_2} ...

    positional arguments:
      {train,predict}

    optional arguments:
      -h, --help           show this help message and exit
      --cfg CFG            if loading config from config file
      --save_cfg SAVE_CFG  save config and exit.

    $ # 运行func_2，config_name1参数覆盖为10，并保存config文件到 config.ini。如果设置--save_cfg，不会真正执行func_2。
    $ main.py --save_cfg config.ini func_2 --config_name1 10

    $ main.py --cfg config.ini func_2 --config_name3 20 # 以config.ini的参数来运行func_2，覆盖 config_name3 为 20

3. 配置的优先级

命令行指定 > 配置文件（若指定） > 程序中的默认参数

4. 更新配置参数

仅限增加配置，而不是修改名字或减少。

这种情况下，请放心在同一个命令中同时使用 `--cfg` 与 `--save_cfg`，甚至设置为同一个文件。这样就可以保证之前的配置继续继承，新的
配置使用程序默认。

"""

import argparse
import codecs
import os
import re
import sys
from configparser import ConfigParser
from os import path
from typing import Union, Dict, List, Any

from .log import log

__all__ = ['add_config', 'inherit_config', 'fire']

MACRO_CONFIG_DIR = 'CONFIG_DIR'
MACRO_CONFIG_FILE = 'CONFIG_FILE'
MACRO_WORKING_DIR = 'WORKING_DIR'


class Singleton(type):
    """
    元类继承type，
    """
    _instance = {}

    def __call__(cls, *args, **kwargs):
        if cls not in cls._instance:
            cls._instance[cls] = super().__call__(*args, **kwargs)
        return cls._instance[cls]


class MacroReplaceManager(object):
    def __init__(self, strict_mod=True, pattern_start='\[\[', pattern_end='\]\]'):
        self.strict_mod = strict_mod
        self.replace_rules = {}
        self.pattern = re.compile(r'(.*)(%s\s*([_a-zA-Z0-9]+)\s*%s)(.*)' % (pattern_start, pattern_end))

    def add_replacement(self, macro: str, replacement: str):
        if self.pattern.match(replacement):
            raise ValueError('%s is illegal, cause matching the pattern %s' % (replacement, self.pattern))
        self.replace_rules[macro] = replacement

    def replace(self, value: str):
        match = self.pattern.match(value)
        if match:
            macro = match.group(3)
            if macro not in self.replace_rules:
                raise ValueError(f'Macro `{macro}` must be set for replace before you get it\'s value.')
            value = match.expand(r'\1%s\4' % self.replace_rules[macro])
            return self.replace(value)
        return value


macro_manager = MacroReplaceManager()


class ConfigItem(object):
    def __init__(self, name: str, dtype: type, default=None, desc: str = '', group=None):
        """

        :param name:
        :param dtype:
        :param default:
        :param desc:
        :param group: default is the class name decorated.
        """
        self.name: str = name
        self.dtype = dtype
        self.default = default
        self.desc: str = desc + ' (default=%(default)s)'
        self.group: str = group

    def __str__(self):
        return 'ConfigItem(name="%s", dtype="%s", default="%.10s...", desc="%s", group="%s")' % (
            self.name, self.dtype, self.default, self.desc, self.group)


class ConfigGroup(object):
    def __init__(self, name: str):
        self.name = name
        self.configs: Dict[str, ConfigItem] = {}
        self.parents: List[ConfigGroup] = []
        self.values: Dict[str, Any] = {}

    def add_parent(self, parent_cls):

        if parent_cls not in self.parents:
            self.parents.append(parent_cls)

    def add_config(self, name: str, dtype: type, default=None, desc: str = ''):
        if dtype not in [int, str, float]:
            raise ValueError('config type only support `int`, `str`, `float`.')
        self.configs[name] = ConfigItem(name, dtype, default, desc, self.name)
        self.values[name] = default

    def set_config(self, name: str, value: Any):
        if name in self.values:
            self.values[name] = self.configs[name].dtype(value)
        else:
            for parent in self.parents:
                if name in parent:
                    parent.set_config(name, value)
                    return
            raise ValueError('unknown config`%s`' % name)

    def get_config(self, name: str, macro_mod=False):
        if name in self.values:
            value = self.values[name]
            if macro_mod and isinstance(value, str):
                return macro_manager.replace(value)
            return value
        else:
            for parent in self.parents:
                if name in parent:
                    return parent.get_config(name, macro_mod)
            raise ValueError('unknown config`%s`' % name)

    def __getitem__(self, item: str):
        return self.get_config(item, macro_mod=True)

    def __contains__(self, item: str):
        return item in self.values or any(item in parent for parent in self.parents)

    def as_dict(self):
        config_dict = {}
        for config_item in self.iter_config_item():
            config_dict[config_item.name] = self.get_config(config_item.name)
        return config_dict

    def __str__(self):
        return self.as_dict().__str__()

    def iter_config_item(self):
        for parent in self.parents:
            for item in parent.iter_config_item():
                yield item
        for item in self.configs.values():
            yield item


class GlobalConfig(metaclass=Singleton):
    def __init__(self):
        self.groups: Dict[str, ConfigGroup] = {}

    def get_config_group(self, group_name: str):
        if group_name not in self.groups:
            self.groups[group_name] = ConfigGroup(group_name)
        return self.groups[group_name]

    def read_from_file(self, cfg_file: str):
        log.info('read config from `%s`' % cfg_file)
        config_parser = ConfigParser()
        config_parser.read(cfg_file, encoding='utf8')
        for group_name in config_parser.sections():
            group = self.get_config_group(group_name)
            for option_name in config_parser.options(group_name):
                group.set_config(option_name, config_parser.get(group_name, option_name))

    def __getitem__(self, item):
        return self.groups.__getitem__(item)


class RunMockClass(object):
    def __init__(self, callable: Union[type]):
        self.name = _get_hack_name(callable)
        self.global_config = GlobalConfig()
        self.callable = callable

    def __set_config(self, key: str, value: Any):
        return self.global_config.get_config_group(self.name).set_config(key, value)

    def __call__(self, *args, **kwargs):
        return self.callable(self.global_config.get_config_group(self.name), *args, **kwargs)


def _get_hack_name(that: object):
    return that.name if hasattr(that, 'name') else that.__name__


def _insert_config(cls_or_fun):
    if isinstance(cls_or_fun, RunMockClass):
        return cls_or_fun
    else:
        callable = RunMockClass(cls_or_fun)
        return callable


# api
def add_config(name: str, dtype: type, default=None, desc: str = ''):
    """
    作为修饰器，给修饰的可调用过程（方法，类）加上一个新的配置项。并在该过程调用时注入第一个参数为config。
    :param name: 配置名称
    :param dtype: 配置的数据类型，仅支持 int, str, float。如果想使用bool，请用int的0和1代替。
    :param default: 默认项，建议设置。如果想表示None，可以用空字符串之类表示。
    :param desc: 描述。help中显示的内容。
    """
    def wrapper(cls):
        cls = _insert_config(cls)
        group_name = _get_hack_name(cls)
        global_config = GlobalConfig()
        global_config.get_config_group(group_name).add_config(name, dtype, default, desc)
        return cls
    return wrapper


# api
def inherit_config(parent_cls):
    """
    作为修饰器，使被修饰的可调用过程继承一套其他的配置。这些配置可在本过程的第一个参数中注入。
    :param parent_cls: 传入其他被 ref:add_config 或 ref:inherit_config 修饰过了的可调用过程。
    """
    def wrapper(cls):
        cls = _insert_config(cls)
        _insert_config(parent_cls)
        parent_group_name = _get_hack_name(parent_cls)
        group_name = _get_hack_name(cls)
        global_config = GlobalConfig()
        global_config.get_config_group(group_name).add_parent(global_config.get_config_group(parent_group_name))
        return cls
    return wrapper


# api
def fire(callables: List[RunMockClass], override_args=None):
    """
    以子命令的方式运行一些可配置的过程。
    :param callables: 一个被 ref:add_config 或 ref:inherit_config 修饰过了的可调用过程的列表。
    :param override_args: default is sys.argv
    :return:

    python 文件中
    -------------

    这些过程必须只有一个必须参数 config.
    每个子命令都会用传入的过程的名字命名。

    命令行运行
    ---------

    python some_script.py [global_config] sub_command [sub_command options]



    1. global_config

    命令行执行时加入了两个参数来解决配置文件（配置的保存和复用）的问题。这两个参数应该在默认参数之前执行。

    ::

        --cfg CONFIG_FILE_PATH  # 设置后会读取配置文件 CONFIG_FILE_PATH 中的配置。覆盖默认配置。
        --save_cfg CONFIG_FILE_PATH # 将默认配置写入 CONFIG_FILE_PATH 。手动修改后再--cfg执行。

    2. sub_command

    与传入函数的每个python过程同名

    3. sub_command options

    该过程通过add_config添加的和其继承的所有配置项。

    """
    args = sys.argv[1:] if override_args is None else override_args
    global_config = GlobalConfig()
    run_ables = callables[:]
    # priority 1: in code default. (in establish the class or function add_config)

    # if loading config from config file
    cfg_file = None
    for arg_id, arg in enumerate(args):
        if arg == '--cfg' and len(args) > arg_id + 2:
            cfg_file = args[arg_id + 1]
            break
        elif arg.startswith('--cfg='):
            cfg_file = arg[arg.find('=') + 1:]
            break

    macro_manager.add_replacement(MACRO_WORKING_DIR, path.abspath('.'))

    if cfg_file is not None:
        macro_manager.add_replacement(MACRO_CONFIG_DIR, path.dirname(cfg_file))
        macro_manager.add_replacement(MACRO_CONFIG_FILE, cfg_file)

        global_config.read_from_file(cfg_file)

    # priority 2: config file if set

    parser = argparse.ArgumentParser()
    # mock config --cfg. only use for --help. config file will be read at priority 1.
    parser.add_argument('--cfg', type=str, default=None, help='if loading config from config file')
    parser.add_argument('--save_cfg', type=str, default=None, help='save config and exit.')

    subparsers = parser.add_subparsers()
    for run_able in run_ables:
        process_name = _get_hack_name(run_able)
        sub_parser = subparsers.add_parser(process_name)
        group = global_config.get_config_group(process_name)
        for config_item in group.iter_config_item():
            sub_parser.add_argument('--%s' % config_item.name,
                                    type=config_item.dtype,
                                    default=group.get_config(config_item.name),
                                    help=config_item.desc)
        sub_parser.set_defaults(process_obj=run_able)
    args = parser.parse_args(args=override_args)

    # override value by command line argument
    if hasattr(args, 'process_obj'):
        launched_group = global_config.get_config_group(_get_hack_name(args.process_obj))
        for config_item in launched_group.iter_config_item():
            launched_group.set_config(config_item.name, getattr(args, config_item.name))

    # priority 3: command line override if set

    # if saving config to file
    if args.save_cfg is not None:
        # make activated to collect the all activate callable and their parents.
        activated = []
        temp_activated = [_get_hack_name(call) for call in run_ables]
        while len(temp_activated):
            group_name = temp_activated.pop(0)
            activated.append(group_name)
            for parent in global_config[group_name].parents:
                if parent.name not in activated and parent.name not in temp_activated:
                    temp_activated.append(parent.name)
        activated.reverse()

        config_parser = ConfigParser()
        for group_name in activated:
            config_parser.add_section(group_name)
            for k, v in global_config.get_config_group(group_name).values.items():
                config_parser.set(group_name, k, str(v))
        if not (os.path.dirname(args.save_cfg) == ''):
            os.makedirs(os.path.dirname(args.save_cfg), exist_ok=True)
        log.info('saving config to `%s` and exit.' % args.save_cfg)
        config_parser.write(codecs.open(args.save_cfg, 'w', encoding='utf8'))
        return

    if not hasattr(args, 'process_obj'):
        parser.print_help()
        exit(0)
    process_obj = args.process_obj
    del args.process_obj
    log.info('launch process `%s`' % _get_hack_name(process_obj))
    log.info('with the configs: \n %s' % global_config.get_config_group(_get_hack_name(process_obj)))
    return process_obj()

